package ga.view.examples.simple;

import ga.core.algorithm.util.RandomSingleton;
import ga.core.individual.IAgeIndividual;
import ga.core.individual.IFitness;
import ga.core.validation.GAContext;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Individual for the simple object example.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public final class SimpleObjectIndividual implements
    IAgeIndividual<SimpleObjectIndividual>, IFitness {
  public static final int TYPE_COUNT = 4;
  public static final int PARAMETERS_COLOR = 3;
  public static final int PARAMETER_TYPE = 3;
  public static final int NUM_PARAMETERS = 4;

  private static final Random RND = RandomSingleton.getRandom();

  private static final AtomicLong ID_GENERATOR = new AtomicLong();

  private final long id;

  private int maxAge;

  private final List<Number> genotype = new ArrayList<Number>();

  private int age;

  private double fitness = UNEVALUATED;

  /**
   * Instantiates a new simple object individual.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public SimpleObjectIndividual() {
    id = ID_GENERATOR.incrementAndGet();
    initRandomly();
  }

  @Override
  public long getId() {
    return id;
  }

  @Override
  public void setContext(final GAContext context) {
  }

  @Override
  public GAContext getContext() {
    return null;
  }

  @Override
  public void initRandomly() {
    fitness = UNEVALUATED;
    genotype.clear();

    for (int i = 0; i < PARAMETERS_COLOR; i++) {
      genotype.add(RND.nextFloat());
    }
    genotype.add(RND.nextInt(4));
  }

  @Override
  public void setFitness(final double fitness) {
    this.fitness = fitness;
  }

  @Override
  public double getFitness() {
    return fitness;
  }

  /**
   * Gets the genotype.
   * 
   * @return the genotype
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public List<Number> getGenotype() {
    return genotype;
  }

  @Override
  public String toString() {
    String s = "ColorRGBA(" + genotype.get(0) + "," + genotype.get(1) + ","
        + genotype.get(2) + " Type: " + genotype.get(PARAMETER_TYPE);

    if (isEvaluated()) {
      s += " Fitness: " + fitness;
    } else {
      s += " unevaluated";
    }

    return s;
  }

  @Override
  public int getAge() {
    return age;
  }

  @Override
  public void setMaxAge(final int maxAge) {
    this.maxAge = maxAge;
  }

  @Override
  public float getAgeNormalized() {
    return ((float) age) / (float) maxAge;
  }

  @Override
  public boolean isOld() {
    return age > maxAge;
  }

  /**
   * Gets the maximum age.
   * 
   * @return the max age
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public int getMaxAge() {
    return maxAge;
  }

  @Override
  public void incAge() {
    this.age++;
  }

  @Override
  public boolean isEvaluated() {
    return fitness != UNEVALUATED;
  }

  @Override
  public SimpleObjectIndividual clone() {
    final SimpleObjectIndividual ind = new SimpleObjectIndividual();
    ind.genotype.addAll(this.genotype);
    ind.setMaxAge(maxAge);
    return ind;
  }

}
